<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\User;
use App\Models\Absensi;
use Carbon\Carbon;
use Maatwebsite\Excel\Facades\Excel;
use Barryvdh\DomPDF\Facade\Pdf; // Tambahkan import ini

class AdminController extends Controller
{
    public function karyawanLepas()
    {
        $karyawanLepas = User::where('role', 'karyawan_lepas')
            ->where('is_active', true)
            ->orderBy('name')
            ->get();
        
        return view('admin.karyawan-lepas', compact('karyawanLepas'));
    }

    public function absenKaryawanLepas(Request $request)
    {
        $request->validate([
            'user_id' => 'required|exists:users,id',
            'tanggal' => 'required|date',
            'jam_masuk' => 'nullable|date_format:H:i',
            'jam_pulang' => 'nullable|date_format:H:i',
            'status' => 'required|in:hadir,izin,sakit,alpha',
            'keterangan' => 'nullable|string|max:500',
        ]);

        $user = User::findOrFail($request->user_id);
        
        // Check if user is karyawan lepas
        if ($user->role !== 'karyawan_lepas') {
            return redirect()->back()->with('error', 'Hanya karyawan lepas yang dapat diabsenkan oleh admin.');
        }

        // Check if attendance already exists for this date
        $existingAbsensi = Absensi::where('user_id', $user->id)
            ->where('tanggal', $request->tanggal)
            ->first();

        if ($existingAbsensi) {
            return redirect()->back()->with('error', 'Absensi untuk tanggal tersebut sudah ada.');
        }

        // Create attendance record
        $absensi = Absensi::create([
            'user_id' => $user->id,
            'tanggal' => $request->tanggal,
            'jam_masuk' => $request->jam_masuk,
            'jam_pulang' => $request->jam_pulang,
            'status' => $request->status,
            'keterangan' => $request->keterangan,
            'tipe_input' => 'admin',
            'admin_id' => Auth::id(),
            'nama_admin' => Auth::user()->name,
            'lokasi_input' => Auth::user()->lokasi,
        ]);

        // Calculate total hours if both check-in and check-out are provided
        if ($request->jam_masuk && $request->jam_pulang) {
            $absensi->hitungTotalJam();
        }

        return redirect()->back()->with('success', 'Absensi berhasil dicatat untuk ' . $user->name);
    }

    public function inputAbsensiLepas(Request $request)
    {
       
        $request->validate([
            'nama_karyawan' => 'required|string|max:255',
            'tanggal' => 'required|date',
            'jam_masuk' => 'required|date_format:H:i',
            'jam_pulang' => 'nullable|date_format:H:i|after:jam_masuk',
            'foto_masuk' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'foto_pulang' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'keterangan' => 'nullable|string|max:500',
        ]);

        // Check if attendance already exists for this name and date
        $existingAbsensi = Absensi::where('nama_karyawan', $request->nama_karyawan)
            ->where('tanggal', $request->tanggal)
            ->first();

        if ($existingAbsensi) {
            return redirect()->back()->with('error', 'Absensi untuk karyawan lepas ini pada tanggal tersebut sudah ada.');
        }

        // Handle photo uploads
        $fotoMasuk = null;
        $fotoPulang = null;

        if ($request->hasFile('foto_masuk')) {
            $fotoMasuk = $request->file('foto_masuk')->store('absensi', 'public');
        }

        if ($request->hasFile('foto_pulang')) {
            $fotoPulang = $request->file('foto_pulang')->store('absensi', 'public');
        }

        // Sebelum create absensi, tambahkan debug berikut:
        // Create attendance record
        $absensi = Absensi::create([
            'user_id' => null,
            'nama_karyawan' => $request->nama_karyawan,
            'tanggal' => $request->tanggal,
            'jam_masuk' => $request->jam_masuk,
            'jam_pulang' => $request->jam_pulang, // boleh null
            'foto_masuk' => $fotoMasuk,
            'foto_pulang' => $fotoPulang,
            'status' => 'hadir',
            'keterangan' => $request->keterangan,
            'tipe_input' => 'admin',
            'admin_id' => Auth::id(),
            'nama_admin' => Auth::user()->name,
            'lokasi_input' => Auth::user()->lokasi,
        ]);

        // Hitung total jam jika jam pulang diisi
        if ($request->jam_masuk && $request->jam_pulang) {
            $absensi->hitungTotalJam();
        }

        return redirect()->back()->with('success', 'Absensi berhasil dicatat untuk ' . $request->nama_karyawan);
    }

    public function editAbsensi($id)
    {
        $absensi = Absensi::with('user')->findOrFail($id);
        
        // Only allow editing admin-input attendance
        if ($absensi->tipe_input !== 'admin') {
            return redirect()->back()->with('error', 'Hanya absensi yang diinput admin yang dapat diedit.');
        }

        return view('admin.edit-absensi', compact('absensi'));
    }

    public function updateAbsensi(Request $request, $id)
    {
        $request->validate([
            'jam_masuk' => 'nullable|date_format:H:i',
            'jam_pulang' => 'nullable|date_format:H:i',
            'status' => 'required|in:hadir,izin,sakit,alpha',
            'keterangan' => 'nullable|string|max:500',
            'foto_masuk' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'foto_pulang' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'lokasi' => 'nullable|string|max:255',
        ]);

        $absensi = Absensi::findOrFail($id);
        
        // Only allow editing admin-input attendance
        if ($absensi->tipe_input !== 'admin') {
            return redirect()->back()->with('error', 'Hanya absensi yang diinput admin yang dapat diedit.');
        }

        $absensi->update([
            'jam_masuk' => $request->jam_masuk,
            'jam_pulang' => $request->jam_pulang,
            'status' => $request->status,
            'keterangan' => $request->keterangan,
            'lokasi' => $request->lokasi,
        ]);

        // Handle new photo uploads
        if ($request->hasFile('foto_masuk')) {
            $fotoMasuk = $request->file('foto_masuk')->store('absensi', 'public');
            $absensi->foto_masuk = $fotoMasuk;
        }
        if ($request->hasFile('foto_pulang')) {
            $fotoPulang = $request->file('foto_pulang')->store('absensi', 'public');
            $absensi->foto_pulang = $fotoPulang;
        }
        $absensi->save();

        // Recalculate total hours if both times are provided
        if ($request->jam_masuk && $request->jam_pulang) {
            $absensi->hitungTotalJam();
        }

        return redirect()->route('admin.absensi-hari-ini')->with('success', 'Absensi berhasil diperbarui.');
    }

    public function deleteAbsensi($id)
    {
        $absensi = Absensi::findOrFail($id);
        
        // Only allow deleting admin-input attendance
        if ($absensi->tipe_input !== 'admin') {
            return redirect()->back()->with('error', 'Hanya absensi yang diinput admin yang dapat dihapus.');
        }

        $absensi->delete();

        return redirect()->back()->with('success', 'Absensi berhasil dihapus.');
    }

    public function absensiHariIni(Request $request)
    {
        $tanggal = $request->input('tanggal', now()->toDateString());
        $absensi = \App\Models\Absensi::with('user')->where('tanggal', $tanggal)->orderBy('jam_masuk')->get();
        
        // Get karyawan lepas who haven't been absent today
        $karyawanLepas = User::where('role', 'karyawan_lepas')
            ->where('is_active', true)
            ->whereNotIn('id', $absensi->pluck('user_id'))
            ->orderBy('name')
            ->get();
        
        return view('admin.absensi-hari-ini', compact('absensi', 'tanggal', 'karyawanLepas'));
    }

    public function absensiMingguan(Request $request)
    {
        $tanggal_awal = $request->input('tanggal_awal', now()->startOfWeek()->toDateString());
        $tanggal_akhir = $request->input('tanggal_akhir', now()->endOfWeek()->toDateString());

        $absensi = Absensi::with('user')
            ->whereBetween('tanggal', [$tanggal_awal, $tanggal_akhir])
            ->orderBy('tanggal', 'desc')
            ->orderBy('jam_masuk')
            ->get();

        return view('admin.absensi-mingguan', compact('absensi', 'tanggal_awal', 'tanggal_akhir'));
    }

    public function hapusAbsensi($id)
    {
        $absensi = \App\Models\Absensi::findOrFail($id);
        $absensi->delete();
        return redirect()->back()->with('success', 'Absensi berhasil dihapus.');
    }

    public function manageUsers()
    {
        $users = User::orderBy('name')->paginate(20);
        
        return view('admin.manage-users', compact('users'));
    }

    public function createUser()
    {
        return view('admin.create-user');
    }

    public function storeUser(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8|confirmed',
            'role' => 'required|in:admin,karyawan_tetap,karyawan_lepas',
            'nip' => 'nullable|string|max:50|unique:users',
            'jabatan' => 'nullable|string|max:255',
            'gaji_pokok' => 'nullable|numeric|min:0',
            'tarif_lembur' => 'nullable|numeric|min:0',
        ]);

        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => bcrypt($request->password),
            'role' => $request->role,
            'nip' => $request->nip,
            'jabatan' => $request->jabatan,
            'gaji_pokok' => $request->gaji_pokok ?? 0,
            'tarif_lembur' => $request->tarif_lembur ?? 0,
            'is_active' => true,
        ]);

        return redirect()->route('admin.dashboard')->with('success', 'User berhasil ditambahkan.');
    }

    public function editUser($id)
    {
        $user = User::findOrFail($id);
        
        return view('admin.edit-user', compact('user'));
    }

    public function updateUser(Request $request, $id)
    {
        $user = User::findOrFail($id);
        
        $validatedData = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users,email,' . $user->id,
            'role' => 'required|in:admin,karyawan_tetap,karyawan_lepas',
            'nip' => 'nullable|string|max:50|unique:users,nip,' . $user->id,
            'jabatan' => 'nullable|string|max:255',
            'gaji_pokok' => 'nullable|numeric|min:0',
            'tarif_lembur' => 'nullable|numeric|min:0',
            'is_active' => 'sometimes|boolean',
        ]);

        $user->name = $validatedData['name'];
        $user->email = $validatedData['email'];
        $user->role = $validatedData['role'];
        $user->nip = $validatedData['nip'] ?? $user->nip;
        $user->jabatan = $validatedData['jabatan'] ?? $user->jabatan;
        $user->gaji_pokok = $validatedData['gaji_pokok'] ?? $user->gaji_pokok;
        $user->tarif_lembur = $validatedData['tarif_lembur'] ?? $user->tarif_lembur;
        $user->is_active = $request->boolean('is_active');
        
        $user->save();
        
        $user->syncRoles([$validatedData['role']]);

        // Re-fetch the user from the database to confirm the role.
        $updatedUser = User::find($user->id);
        \Illuminate\Support\Facades\Log::info("User {$updatedUser->id} role after update: {$updatedUser->role}");

        return redirect()->route('admin.manage-users')->with('success', 'User berhasil diperbarui. Role: ' . $updatedUser->role);
    }

    public function exportPDF(Request $request)
    {
        $tanggal = $request->input('tanggal', now()->toDateString());
        $absensi = Absensi::with('user')
            ->where('tanggal', $tanggal)
            ->orderBy('jam_masuk')
            ->get();

        $pdf = PDF::loadView('admin.exports.absensi-pdf', [
            'absensi' => $absensi,
            'tanggal' => $tanggal
        ]);

        return $pdf->download('absensi-' . $tanggal . '.pdf');
    }

    public function exportExcel(Request $request)
    {
        $tanggal = $request->input('tanggal', now()->toDateString());
        $absensi = Absensi::with('user')
            ->where('tanggal', $tanggal)
            ->orderBy('jam_masuk')
            ->get();

        return Excel::download(new \App\Exports\AbsensiExport($absensi), 'absensi-' . $tanggal . '.xlsx');
    }

    public function exportPDFMingguan(Request $request)
    {
        $tanggal_awal = $request->input('tanggal_awal', now()->startOfWeek()->toDateString());
        $tanggal_akhir = $request->input('tanggal_akhir', now()->endOfWeek()->toDateString());

        $absensi = Absensi::with('user')
            ->whereBetween('tanggal', [$tanggal_awal, $tanggal_akhir])
            ->orderBy('tanggal', 'desc')
            ->orderBy('jam_masuk')
            ->get();

        $pdf = PDF::loadView('admin.exports.absensi-mingguan-pdf', [
            'absensi' => $absensi,
            'tanggal_awal' => $tanggal_awal,
            'tanggal_akhir' => $tanggal_akhir
        ]);

        return $pdf->download('absensi_' . $tanggal_awal . '_' . $tanggal_akhir . '.pdf');
    }

    public function exportExcelMingguan(Request $request)
    {
        $tanggal_awal = $request->input('tanggal_awal', now()->startOfWeek()->toDateString());
        $tanggal_akhir = $request->input('tanggal_akhir', now()->endOfWeek()->toDateString());

        $absensi = Absensi::with('user')
            ->whereBetween('tanggal', [$tanggal_awal, $tanggal_akhir])
            ->orderBy('tanggal', 'desc')
            ->orderBy('jam_masuk')
            ->get();

        return Excel::download(new \App\Exports\AbsensiExport($absensi), 'absensi_' . $tanggal_awal . '_' . $tanggal_akhir . '.xlsx');
    }
}
