<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Absensi;
use Carbon\Carbon;

class AbsensiController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        $absensiHariIni = $user->getAbsensiHariIni();
        
        return view('karyawan.absensi', compact('absensiHariIni'));
    }

    public function masuk(Request $request)
    {
        $user = Auth::user();
        $today = Carbon::today();

        $request->validate([
            'foto_masuk' => 'required|image|mimes:jpeg,png,jpg|max:2048',
        ]);

        // Check if already checked in today
        $existingAbsensi = $user->getAbsensiHariIni();
        if ($existingAbsensi && $existingAbsensi->jam_masuk) {
            return redirect()->back()->with('error', 'Anda sudah melakukan absen masuk hari ini.');
        }

        // Upload foto masuk
        $fotoPath = $request->file('foto_masuk')->store('absensi/foto_masuk', 'public');

        // Create or update attendance record
        $absensi = Absensi::updateOrCreate(
            [
                'user_id' => $user->id,
                'tanggal' => $today,
            ],
            [
                'jam_masuk' => Carbon::now(),
                'foto_masuk' => $fotoPath,
                'status' => 'hadir',
                'tipe_input' => 'mandiri',
                'lokasi' => $user->lokasi,
            ]
        );

        return redirect()->back()->with('success', 'Absen masuk berhasil dicatat pada ' . Carbon::now()->format('H:i:s'));
    }

    public function pulang(Request $request)
    {
        $user = Auth::user();
        $today = Carbon::today();

        $request->validate([
            'foto_pulang' => 'required|image|mimes:jpeg,png,jpg|max:2048',
        ]);

        // Get today's attendance
        $absensi = $user->getAbsensiHariIni();

        if (!$absensi) {
            return redirect()->back()->with('error', 'Anda belum melakukan absen masuk hari ini.');
        }

        if ($absensi->jam_pulang) {
            return redirect()->back()->with('error', 'Anda sudah melakukan absen pulang hari ini.');
        }

        // Upload foto pulang
        $fotoPath = $request->file('foto_pulang')->store('absensi/foto_pulang', 'public');

        // Update check-out time
        $absensi->jam_pulang = Carbon::now();
        $absensi->foto_pulang = $fotoPath;
        $absensi->lokasi = $user->lokasi;
        $absensi->save();

        // Calculate total hours and overtime
        $absensi->hitungTotalJam();

        return redirect()->back()->with('success', 'Absen pulang berhasil dicatat pada ' . Carbon::now()->format('H:i:s'));
    }

    public function izin(Request $request)
    {
        $request->validate([
            'keterangan' => 'required|string|max:500',
        ]);
        
        $user = Auth::user();
        $today = Carbon::today();
        
        // Check if already has attendance record today
        $existingAbsensi = $user->getAbsensiHariIni();
        if ($existingAbsensi) {
            return redirect()->back()->with('error', 'Anda sudah memiliki catatan absensi hari ini.');
        }
        
        // Create attendance record for leave
        $absensi = Absensi::create([
            'user_id' => $user->id,
            'tanggal' => $today,
            'status' => 'izin',
            'keterangan' => $request->keterangan,
            'tipe_input' => 'mandiri',
        ]);
        
        return redirect()->back()->with('success', 'Pengajuan izin berhasil dicatat.');
    }

    public function sakit(Request $request)
    {
        $request->validate([
            'keterangan' => 'required|string|max:500',
        ]);
        
        $user = Auth::user();
        $today = Carbon::today();
        
        // Check if already has attendance record today
        $existingAbsensi = $user->getAbsensiHariIni();
        if ($existingAbsensi) {
            return redirect()->back()->with('error', 'Anda sudah memiliki catatan absensi hari ini.');
        }
        
        // Create attendance record for sick leave
        $absensi = Absensi::create([
            'user_id' => $user->id,
            'tanggal' => $today,
            'status' => 'sakit',
            'keterangan' => $request->keterangan,
            'tipe_input' => 'mandiri',
        ]);
        
        return redirect()->back()->with('success', 'Pengajuan sakit berhasil dicatat.');
    }
}
